#!/usr/bin/env php
<?php
/**
 * WebSocket Server Manager Script
 * 
 * This script is used to start, stop, and manage the WebSocket server
 * 
 * Usage:
 * php websocket_manager.php start    - Start the WebSocket server
 * php websocket_manager.php stop     - Stop the WebSocket server
 * php websocket_manager.php restart  - Restart the WebSocket server
 * php websocket_manager.php status   - Check the status of the WebSocket server
 */

// Define the path to the WebSocket server script
$websocket_server_script = __DIR__ . '/websocket_server.php';

// Define the path to the PID file
$pid_file = __DIR__ . '/websocket_server.pid';

// Define the log file
$log_file = __DIR__ . '/websocket_server.log';

// Check if the command argument is provided
if ($argc < 2) {
    echo "Usage: php websocket_manager.php [start|stop|restart|status]\n";
    exit(1);
}

$command = $argv[1];

// Function to check if the server is running
function is_server_running() {
    global $pid_file;
    
    if (!file_exists($pid_file)) {
        return false;
    }
    
    $pid = file_get_contents($pid_file);
    
    // Check if the process exists
    if (posix_kill($pid, 0)) {
        return $pid;
    } else {
        // Remove the PID file if the process doesn't exist
        unlink($pid_file);
        return false;
    }
}

// Function to start the server
function start_server() {
    global $websocket_server_script, $pid_file, $log_file;
    
    if (is_server_running()) {
        echo "WebSocket server is already running.\n";
        return;
    }
    
    echo "Starting WebSocket server...\n";
    
    // Start the server as a background process
    $command = sprintf(
        'nohup php %s > %s 2>&1 & echo $! > %s',
        $websocket_server_script,
        $log_file,
        $pid_file
    );
    
    exec($command);
    
    // Wait a moment to ensure the server has started
    sleep(1);
    
    if (is_server_running()) {
        echo "WebSocket server started successfully.\n";
    } else {
        echo "Failed to start WebSocket server. Check the log file: $log_file\n";
    }
}

// Function to stop the server
function stop_server() {
    global $pid_file;
    
    $pid = is_server_running();
    
    if (!$pid) {
        echo "WebSocket server is not running.\n";
        return;
    }
    
    echo "Stopping WebSocket server...\n";
    
    // Send a SIGTERM signal to the process
    posix_kill($pid, SIGTERM);
    
    // Wait a moment to ensure the server has stopped
    sleep(1);
    
    if (is_server_running()) {
        echo "Failed to stop WebSocket server. Trying SIGKILL...\n";
        posix_kill($pid, SIGKILL);
        
        // Wait again
        sleep(1);
        
        if (is_server_running()) {
            echo "Failed to stop WebSocket server. Please check the process manually.\n";
        } else {
            echo "WebSocket server stopped successfully.\n";
            
            // Remove the PID file
            if (file_exists($pid_file)) {
                unlink($pid_file);
            }
        }
    } else {
        echo "WebSocket server stopped successfully.\n";
        
        // Remove the PID file
        if (file_exists($pid_file)) {
            unlink($pid_file);
        }
    }
}

// Function to restart the server
function restart_server() {
    stop_server();
    start_server();
}

// Function to check the status of the server
function server_status() {
    $pid = is_server_running();
    
    if ($pid) {
        echo "WebSocket server is running with PID: $pid\n";
    } else {
        echo "WebSocket server is not running.\n";
    }
}

// Handle the command
switch ($command) {
    case 'start':
        start_server();
        break;
    
    case 'stop':
        stop_server();
        break;
    
    case 'restart':
        restart_server();
        break;
    
    case 'status':
        server_status();
        break;
    
    default:
        echo "Unknown command: $command\n";
        echo "Usage: php websocket_manager.php [start|stop|restart|status]\n";
        exit(1);
}

exit(0);
