<?php
/**
 * KSO Chandigarh URL Initialization File
 * 
 * This file ensures that URLs work correctly with or without file extensions.
 * It should be included at the top of all public-facing PHP files.
 */

// Define constants for URL paths
define('BASE_URL', 'http://localhost:8000');
define('ADMIN_URL', BASE_URL . '/adminlte');

/**
 * Function to convert a URL with a PHP extension to a clean URL
 * 
 * @param string $url The URL with .php extension
 * @return string The clean URL without .php extension
 */
function cleanUrl($url) {
    return str_replace('.php', '', $url);
}

/**
 * Get the current page URL without the .php extension
 * 
 * @return string The current page URL
 */
function getCurrentCleanUrl() {
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $uri = $_SERVER['REQUEST_URI'];
    
    // Remove .php extension if present
    $uri = preg_replace('/\.php([\?#].*)?$/', '$1', $uri);
    
    return $protocol . '://' . $host . $uri;
}

/**
 * Check if the current server environment supports URL rewriting
 * 
 * @return bool True if URL rewriting is supported
 */
function isRewriteEnabled() {
    // Create a test file to check if rewriting works
    $testFile = __DIR__ . '/rewrite_test.tmp';
    $testUrl = BASE_URL . '/rewrite_test';
    
    // Create the test file
    file_put_contents($testFile, 'URL rewrite test');
    
    // Try to access the URL without extension
    $ch = curl_init($testUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HEADER, false);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    // Clean up
    @unlink($testFile);
    
    // If we got a 200 response, rewriting works
    return $httpCode === 200;
}

/**
 * Update links in HTML to use clean URLs
 * 
 * @param string $html The HTML content
 * @return string Updated HTML with clean URLs
 */
function convertHtmlToCleanUrls($html) {
    // Replace links to PHP files
    $html = preg_replace('/href=(["\'])(.*?)\.php(["\'])/', 'href=$1$2$3', $html);
    
    return $html;
}

/**
 * Add canonical link to HTML head
 * 
 * @param string $url The canonical URL
 * @return string HTML for canonical link
 */
function getCanonicalLink($url = null) {
    if ($url === null) {
        $url = getCurrentCleanUrl();
    }
    
    return '<link rel="canonical" href="' . htmlspecialchars($url) . '" />';
}

/**
 * Initialize output buffering to automatically convert URLs in the output
 */
function initCleanUrls() {
    ob_start(function($buffer) {
        return convertHtmlToCleanUrls($buffer);
    });
}

// Automatically start clean URLs processing if requested
if (isset($_GET['clean_urls']) && $_GET['clean_urls'] === 'auto') {
    initCleanUrls();
}
