<?php
/**
 * Update PHP Files to Include Initialization Script
 * 
 * This script adds the include statement for init.php to all PHP files
 * in the specified directories to enable clean URLs.
 */

// Define the directories to process
$directories = [
    __DIR__, // Main directory
    __DIR__ . '/adminlte' // AdminLTE directory
];

// Files to exclude from processing
$excludeFiles = [
    'init.php',
    'config/database.php',
    'config/auth_enhanced.php',
    'config/email_config.php',
    'api/', // Skip API files
    'services/', // Skip service files
    'vendor/', // Skip vendor files
    'update_files.php' // Skip this file
];

// Define the include statement to add
$mainInclude = "// Include initialization file for clean URLs\nrequire_once __DIR__ . '/init.php';\n\n";
$adminInclude = "// Include initialization file for clean URLs\nrequire_once 'init.php';\n\n";

// Counter for processed files
$processed = 0;
$updated = 0;
$skipped = 0;

/**
 * Check if a file should be excluded
 * 
 * @param string $filePath The file path to check
 * @return bool True if the file should be excluded
 */
function shouldExclude($filePath) {
    global $excludeFiles;
    
    foreach ($excludeFiles as $exclude) {
        if (strpos($filePath, $exclude) !== false) {
            return true;
        }
    }
    
    return false;
}

/**
 * Process a PHP file to add the include statement
 * 
 * @param string $filePath The file path to process
 * @return bool True if the file was updated
 */
function processFile($filePath) {
    global $mainInclude, $adminInclude, $processed, $updated, $skipped;
    
    // Skip files that should be excluded
    if (shouldExclude($filePath)) {
        echo "Skipping: $filePath\n";
        $skipped++;
        return false;
    }
    
    // Read the file content
    $content = file_get_contents($filePath);
    
    // Skip if it already has the include
    if (strpos($content, 'require_once') !== false && 
        (strpos($content, 'init.php') !== false || strpos($content, '/init.php') !== false)) {
        echo "Already updated: $filePath\n";
        $skipped++;
        return false;
    }
    
    // Skip files that don't have the PHP opening tag
    if (strpos($content, '<?php') === false) {
        echo "Not a PHP file with opening tag: $filePath\n";
        $skipped++;
        return false;
    }
    
    // Replace the PHP opening tag with the tag plus the include statement
    $isAdminFile = strpos($filePath, '/adminlte/') !== false;
    $include = $isAdminFile ? $adminInclude : $mainInclude;
    
    $updatedContent = str_replace('<?php', '<?php' . "\n" . $include, $content);
    
    // Write the updated content back to the file
    if (file_put_contents($filePath, $updatedContent)) {
        echo "Updated: $filePath\n";
        $updated++;
        return true;
    }
    
    return false;
}

/**
 * Process all PHP files in a directory recursively
 * 
 * @param string $directory The directory to process
 */
function processDirectory($directory) {
    global $processed;
    
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($directory, RecursiveDirectoryIterator::SKIP_DOTS)
    );
    
    foreach ($iterator as $file) {
        if ($file->isFile() && $file->getExtension() === 'php') {
            $processed++;
            processFile($file->getPathname());
        }
    }
}

// Process each directory
foreach ($directories as $directory) {
    echo "Processing directory: $directory\n";
    processDirectory($directory);
}

// Output summary
echo "\nSummary:\n";
echo "Files processed: $processed\n";
echo "Files updated: $updated\n";
echo "Files skipped: $skipped\n";
