<?php
/**
 * KSO Chandigarh - Password Reset Utility
 * This utility allows resetting passwords for admin users by username or KSO ID
 */

// Database configuration
define('DB_HOST', 'localhost');
define('DB_NAME', 'ktxmmosu_kso_chandigarh');
define('DB_USER', 'ktxmmosu_kso_db');
define('DB_PASS', 'Admin@123');

// Initialize database connection
try {
    $pdo = new PDO(
        "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=utf8mb4",
        DB_USER,
        DB_PASS,
        [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]
    );
} catch (PDOException $e) {
    die("Database connection failed: " . $e->getMessage());
}

$message = '';
$error = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $identifier = trim($_POST['identifier'] ?? '');
    $new_password = trim($_POST['new_password'] ?? '');
    $confirm_password = trim($_POST['confirm_password'] ?? '');
    
    // Validate inputs
    if (empty($identifier)) {
        $error = "Please enter a username or KSO ID";
    } elseif (empty($new_password)) {
        $error = "Please enter a new password";
    } elseif ($new_password !== $confirm_password) {
        $error = "Passwords do not match";
    } elseif (strlen($new_password) < 6) {
        $error = "Password must be at least 6 characters long";
    } else {
        try {
            // Search for admin by username or KSO ID
            $stmt = $pdo->prepare("
                SELECT id, username, kso_id, first_name, last_name, email, role 
                FROM admins 
                WHERE username = ? OR kso_id = ?
                LIMIT 1
            ");
            $stmt->execute([$identifier, $identifier]);
            $admin = $stmt->fetch();
            
            if ($admin) {
                // Hash the new password
                $password_hash = password_hash($new_password, PASSWORD_DEFAULT);
                
                // Update the password
                $update_stmt = $pdo->prepare("
                    UPDATE admins 
                    SET password_hash = ?, updated_at = NOW() 
                    WHERE id = ?
                ");
                $update_stmt->execute([$password_hash, $admin['id']]);
                
                // Log the activity
                $log_stmt = $pdo->prepare("
                    INSERT INTO activity_logs (user_id, action, details, ip_address, created_at) 
                    VALUES (?, 'password_reset', ?, ?, NOW())
                ");
                $log_stmt->execute([
                    $admin['id'], 
                    "Password reset via utility for: {$admin['username']}", 
                    $_SERVER['REMOTE_ADDR'] ?? 'unknown'
                ]);
                
                $message = "Password successfully updated for:<br>
                          <strong>Name:</strong> {$admin['first_name']} {$admin['last_name']}<br>
                          <strong>Username:</strong> {$admin['username']}<br>
                          <strong>KSO ID:</strong> {$admin['kso_id']}<br>
                          <strong>Role:</strong> {$admin['role']}<br>
                          <strong>Email:</strong> {$admin['email']}";
            } else {
                $error = "No admin found with username or KSO ID: $identifier";
            }
        } catch (Exception $e) {
            $error = "Error updating password: " . $e->getMessage();
        }
    }
}

// Get all admins for reference
try {
    $stmt = $pdo->prepare("
        SELECT id, username, kso_id, first_name, last_name, role, email, status 
        FROM admins 
        ORDER BY role, first_name
    ");
    $stmt->execute();
    $all_admins = $stmt->fetchAll();
} catch (Exception $e) {
    $all_admins = [];
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>KSO Chandigarh - Password Reset Utility</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
            background-color: #f5f5f5;
        }
        .container {
            background: white;
            padding: 30px;
            border-radius: 10px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            text-align: center;
            border-bottom: 3px solid #3498db;
            padding-bottom: 10px;
        }
        .form-group {
            margin-bottom: 20px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
            color: #34495e;
        }
        input[type="text"], input[type="password"] {
            width: 100%;
            padding: 10px;
            border: 2px solid #ddd;
            border-radius: 5px;
            font-size: 16px;
            box-sizing: border-box;
        }
        input[type="text"]:focus, input[type="password"]:focus {
            border-color: #3498db;
            outline: none;
        }
        button {
            background-color: #3498db;
            color: white;
            padding: 12px 30px;
            border: none;
            border-radius: 5px;
            font-size: 16px;
            cursor: pointer;
            width: 100%;
        }
        button:hover {
            background-color: #2980b9;
        }
        .success {
            background-color: #d4edda;
            color: #155724;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border: 1px solid #c3e6cb;
        }
        .error {
            background-color: #f8d7da;
            color: #721c24;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
            border: 1px solid #f5c6cb;
        }
        .admin-list {
            margin-top: 30px;
            padding-top: 20px;
            border-top: 2px solid #eee;
        }
        .admin-item {
            background-color: #f8f9fa;
            padding: 15px;
            margin-bottom: 10px;
            border-radius: 5px;
            border-left: 4px solid #3498db;
        }
        .admin-item strong {
            color: #2c3e50;
        }
        .status-active {
            color: #27ae60;
            font-weight: bold;
        }
        .status-inactive {
            color: #e74c3c;
            font-weight: bold;
        }
        .help-text {
            font-size: 14px;
            color: #7f8c8d;
            margin-top: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔐 KSO Chandigarh - Password Reset Utility</h1>
        
        <?php if ($message): ?>
            <div class="success"><?php echo $message; ?></div>
        <?php endif; ?>
        
        <?php if ($error): ?>
            <div class="error"><?php echo $error; ?></div>
        <?php endif; ?>
        
        <form method="POST">
            <div class="form-group">
                <label for="identifier">Username or KSO ID:</label>
                <input type="text" id="identifier" name="identifier" required 
                       placeholder="Enter username (e.g., haominthang) or KSO ID (e.g., KSOCHD001)"
                       value="<?php echo htmlspecialchars($_POST['identifier'] ?? ''); ?>">
                <div class="help-text">You can search by either username or KSO ID</div>
            </div>
            
            <div class="form-group">
                <label for="new_password">New Password:</label>
                <input type="password" id="new_password" name="new_password" required 
                       placeholder="Enter new password (minimum 6 characters)">
                <div class="help-text">Minimum 6 characters required</div>
            </div>
            
            <div class="form-group">
                <label for="confirm_password">Confirm Password:</label>
                <input type="password" id="confirm_password" name="confirm_password" required 
                       placeholder="Re-enter the new password">
            </div>
            
            <button type="submit">🔄 Reset Password</button>
        </form>
        
        <div class="admin-list">
            <h2>📋 Available Admin Accounts</h2>
            <p style="color: #7f8c8d; margin-bottom: 20px;">
                You can use any of the following usernames or KSO IDs to reset passwords:
            </p>
            
            <?php foreach ($all_admins as $admin): ?>
                <div class="admin-item">
                    <strong><?php echo htmlspecialchars($admin['first_name'] . ' ' . $admin['last_name']); ?></strong>
                    <span class="status-<?php echo $admin['status']; ?>">(<?php echo ucfirst($admin['status']); ?>)</span>
                    <br>
                    <strong>Username:</strong> <?php echo htmlspecialchars($admin['username']); ?> | 
                    <strong>KSO ID:</strong> <?php echo htmlspecialchars($admin['kso_id']); ?> | 
                    <strong>Role:</strong> <?php echo htmlspecialchars($admin['role']); ?>
                    <br>
                    <strong>Email:</strong> <?php echo htmlspecialchars($admin['email']); ?>
                </div>
            <?php endforeach; ?>
        </div>
        
        <div style="margin-top: 30px; padding: 20px; background-color: #fff3cd; border-radius: 5px; border: 1px solid #ffeaa7;">
            <h3>🔒 Security Notice</h3>
            <p><strong>Important:</strong> This is a password reset utility for administrative purposes.</p>
            <ul>
                <li>All password changes are logged in the activity logs</li>
                <li>Use strong passwords for security</li>
                <li>Remove or restrict access to this file after use</li>
                <li>Only authorized personnel should use this utility</li>
            </ul>
        </div>
        
        <div style="text-align: center; margin-top: 20px; color: #7f8c8d;">
            <p>KSO Chandigarh Management System | Password Reset Utility</p>
            <p><a href="adminlte/login.php" style="color: #3498db;">← Back to Login</a></p>
        </div>
    </div>
</body>
</html>
