/**
 * KSO Chandigarh Toast Notification System
 * Advanced notification system with multiple types and positions
 */

class KSOToast {
    constructor() {
        this.container = null;
        this.toasts = [];
        this.defaultOptions = {
            position: 'top-right',
            duration: 5000,
            closable: true,
            progress: true,
            sound: false,
            animation: 'slide'
        };
        this.init();
    }

    init() {
        this.createContainer();
        this.addStyles();
    }

    createContainer() {
        this.container = document.createElement('div');
        this.container.id = 'kso-toast-container';
        this.container.className = 'toast-container position-fixed';
        this.setPosition('top-right');
        document.body.appendChild(this.container);
    }

    addStyles() {
        const style = document.createElement('style');
        style.textContent = `
            .toast-container {
                z-index: 9999;
                max-width: 350px;
                padding: 15px;
            }

            .toast-container.top-right {
                top: 20px;
                right: 20px;
            }

            .toast-container.top-left {
                top: 20px;
                left: 20px;
            }

            .toast-container.bottom-right {
                bottom: 20px;
                right: 20px;
            }

            .toast-container.bottom-left {
                bottom: 20px;
                left: 20px;
            }

            .toast-container.top-center {
                top: 20px;
                left: 50%;
                transform: translateX(-50%);
            }

            .toast-container.bottom-center {
                bottom: 20px;
                left: 50%;
                transform: translateX(-50%);
            }

            .kso-toast {
                display: flex;
                align-items: flex-start;
                background: #fff;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                margin-bottom: 10px;
                padding: 16px;
                border-left: 4px solid #007bff;
                position: relative;
                overflow: hidden;
                transition: all 0.3s ease;
                min-height: 60px;
            }

            .kso-toast.success {
                border-left-color: #28a745;
            }

            .kso-toast.error {
                border-left-color: #dc3545;
            }

            .kso-toast.warning {
                border-left-color: #ffc107;
            }

            .kso-toast.info {
                border-left-color: #17a2b8;
            }

            .kso-toast-icon {
                margin-right: 12px;
                font-size: 20px;
                margin-top: 2px;
                flex-shrink: 0;
            }

            .kso-toast-icon.success {
                color: #28a745;
            }

            .kso-toast-icon.error {
                color: #dc3545;
            }

            .kso-toast-icon.warning {
                color: #ffc107;
            }

            .kso-toast-icon.info {
                color: #17a2b8;
            }

            .kso-toast-content {
                flex: 1;
            }

            .kso-toast-title {
                font-weight: 600;
                margin-bottom: 4px;
                color: #333;
                font-size: 14px;
            }

            .kso-toast-message {
                color: #666;
                font-size: 13px;
                line-height: 1.4;
                margin: 0;
            }

            .kso-toast-close {
                background: none;
                border: none;
                font-size: 16px;
                color: #999;
                cursor: pointer;
                padding: 0;
                margin-left: 12px;
                flex-shrink: 0;
                transition: color 0.2s ease;
            }

            .kso-toast-close:hover {
                color: #333;
            }

            .kso-toast-progress {
                position: absolute;
                bottom: 0;
                left: 0;
                height: 3px;
                background: rgba(0, 0, 0, 0.1);
                width: 100%;
                overflow: hidden;
            }

            .kso-toast-progress-bar {
                height: 100%;
                background: #007bff;
                width: 100%;
                transition: width linear;
            }

            .kso-toast-progress-bar.success {
                background: #28a745;
            }

            .kso-toast-progress-bar.error {
                background: #dc3545;
            }

            .kso-toast-progress-bar.warning {
                background: #ffc107;
            }

            .kso-toast-progress-bar.info {
                background: #17a2b8;
            }

            /* Animations */
            @keyframes slideInRight {
                from {
                    transform: translateX(100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }

            @keyframes slideOutRight {
                from {
                    transform: translateX(0);
                    opacity: 1;
                }
                to {
                    transform: translateX(100%);
                    opacity: 0;
                }
            }

            @keyframes slideInLeft {
                from {
                    transform: translateX(-100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }

            @keyframes slideOutLeft {
                from {
                    transform: translateX(0);
                    opacity: 1;
                }
                to {
                    transform: translateX(-100%);
                    opacity: 0;
                }
            }

            @keyframes fadeIn {
                from {
                    opacity: 0;
                    transform: translateY(-20px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }

            @keyframes fadeOut {
                from {
                    opacity: 1;
                    transform: translateY(0);
                }
                to {
                    opacity: 0;
                    transform: translateY(-20px);
                }
            }

            .kso-toast.slide-in-right {
                animation: slideInRight 0.3s ease;
            }

            .kso-toast.slide-out-right {
                animation: slideOutRight 0.3s ease;
            }

            .kso-toast.slide-in-left {
                animation: slideInLeft 0.3s ease;
            }

            .kso-toast.slide-out-left {
                animation: slideOutLeft 0.3s ease;
            }

            .kso-toast.fade-in {
                animation: fadeIn 0.3s ease;
            }

            .kso-toast.fade-out {
                animation: fadeOut 0.3s ease;
            }

            /* Dark mode */
            .dark-mode .kso-toast {
                background: #2c3e50;
                color: #ecf0f1;
            }

            .dark-mode .kso-toast-title {
                color: #ecf0f1;
            }

            .dark-mode .kso-toast-message {
                color: #bdc3c7;
            }

            .dark-mode .kso-toast-close {
                color: #95a5a6;
            }

            .dark-mode .kso-toast-close:hover {
                color: #ecf0f1;
            }
        `;
        document.head.appendChild(style);
    }

    setPosition(position) {
        this.container.className = `toast-container position-fixed ${position}`;
    }

    show(message, type = 'info', options = {}) {
        const opts = { ...this.defaultOptions, ...options };
        
        const toast = this.createToast(message, type, opts);
        this.container.appendChild(toast);
        this.toasts.push(toast);

        // Animate in
        this.animateIn(toast, opts.animation);

        // Auto remove
        if (opts.duration > 0) {
            const progressBar = toast.querySelector('.kso-toast-progress-bar');
            if (progressBar) {
                progressBar.style.transition = `width ${opts.duration}ms linear`;
                progressBar.style.width = '0%';
            }

            setTimeout(() => {
                this.remove(toast, opts.animation);
            }, opts.duration);
        }

        return toast;
    }

    createToast(message, type, options) {
        const toast = document.createElement('div');
        toast.className = `kso-toast ${type}`;
        
        const icons = {
            success: 'fas fa-check-circle',
            error: 'fas fa-exclamation-circle',
            warning: 'fas fa-exclamation-triangle',
            info: 'fas fa-info-circle'
        };

        const titles = {
            success: 'Success',
            error: 'Error',
            warning: 'Warning',
            info: 'Information'
        };

        const closeButton = options.closable ? `
            <button class="kso-toast-close" onclick="window.ksoToast.remove(this.closest('.kso-toast'))">&times;</button>
        ` : '';

        const progressBar = options.progress ? `
            <div class="kso-toast-progress">
                <div class="kso-toast-progress-bar ${type}"></div>
            </div>
        ` : '';

        toast.innerHTML = `
            <div class="kso-toast-icon ${type}">
                <i class="${icons[type]}"></i>
            </div>
            <div class="kso-toast-content">
                <div class="kso-toast-title">${options.title || titles[type]}</div>
                <div class="kso-toast-message">${message}</div>
            </div>
            ${closeButton}
            ${progressBar}
        `;

        return toast;
    }

    animateIn(toast, animation) {
        const position = this.container.classList.contains('top-right') || 
                        this.container.classList.contains('bottom-right') ? 'right' : 'left';
        
        switch (animation) {
            case 'slide':
                toast.classList.add(`slide-in-${position}`);
                break;
            case 'fade':
                toast.classList.add('fade-in');
                break;
            default:
                toast.classList.add('fade-in');
        }
    }

    animateOut(toast, animation) {
        const position = this.container.classList.contains('top-right') || 
                        this.container.classList.contains('bottom-right') ? 'right' : 'left';
        
        switch (animation) {
            case 'slide':
                toast.classList.add(`slide-out-${position}`);
                break;
            case 'fade':
                toast.classList.add('fade-out');
                break;
            default:
                toast.classList.add('fade-out');
        }
    }

    remove(toast, animation = 'fade') {
        this.animateOut(toast, animation);
        
        setTimeout(() => {
            if (toast.parentNode) {
                toast.parentNode.removeChild(toast);
            }
            this.toasts = this.toasts.filter(t => t !== toast);
        }, 300);
    }

    success(message, options = {}) {
        return this.show(message, 'success', options);
    }

    error(message, options = {}) {
        return this.show(message, 'error', options);
    }

    warning(message, options = {}) {
        return this.show(message, 'warning', options);
    }

    info(message, options = {}) {
        return this.show(message, 'info', options);
    }

    clear() {
        this.toasts.forEach(toast => this.remove(toast));
    }
}

// Initialize toast system
document.addEventListener('DOMContentLoaded', () => {
    window.ksoToast = new KSOToast();
});
